/*****************************************************************************
 * Copyright (c) 2022, Nsing Technologies Pte. Ltd.
 *
 * All rights reserved.
 * ****************************************************************************
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer below.
 *
 * Nsing' name may not be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * DISCLAIMER: THIS SOFTWARE IS PROVIDED BY Nsing "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * DISCLAIMED. IN NO EVENT SHALL Nsing BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ****************************************************************************/

/**
 * @file main.c
 * @author Nsing
 * @version V1.2.2
 *
 * @copyright Copyright (c) 2022, Nsing Technologies Pte. Ltd. All rights reserved.
 */
#include "main.h"
#include <stdio.h>
#include <stdint.h>
#include "n32l40x.h"
#include "log.h"
#include "User_Sysclock_Config.h"
#include "User_LED_Config.h"

ADC_InitType ADC_InitStructure;
__IO uint16_t ADCConvertedValue[5];

/** @addtogroup PWR_STOP
 * @{
 */

/**
 * @brief  Delay function.
 *     @arg nCount
 */
void delay(vu32 nCount)
{
    vu32 index = 0;
    for (index = (34000 * nCount); index != 0; index--)
    {
    }
}

void Delay_ms(uint32_t nCount)
{
    uint32_t tcnt;
    while (nCount--)
    {
        tcnt = 64000 / 5;
        while (tcnt--)
        {
            ;
        }
    }
}
void ADC_Initial(void)
{
    NVIC_InitType stNvicInit;

    ADC_DeInit(ADC);
    /* ADC configuration ------------------------------------------------------*/
    ADC_InitStructure.MultiChEn = DISABLE;
    ADC_InitStructure.ContinueConvEn = DISABLE;
    ADC_InitStructure.ExtTrigSelect = ADC_EXT_TRIGCONV_NONE;
    ADC_InitStructure.DatAlign = ADC_DAT_ALIGN_R;
    ADC_InitStructure.ChsNumber = 1;
    ADC_Init(ADC, &ADC_InitStructure);

    /* Configure ADC interrupt */
    stNvicInit.NVIC_IRQChannel = ADC_IRQn;
    stNvicInit.NVIC_IRQChannelPreemptionPriority = 0;
    stNvicInit.NVIC_IRQChannelSubPriority = 0;
    stNvicInit.NVIC_IRQChannelCmd = ENABLE;
    NVIC_Init(&stNvicInit);

    /* Enable ENDC interrupt */
    ADC_ConfigInt(ADC, ADC_INT_ENDC, ENABLE);

    /* Enable ADC */
    ADC_Enable(ADC, ENABLE);
    /* Check ADC Ready */
    while (ADC_GetFlagStatusNew(ADC, ADC_FLAG_RDY) == RESET)
        ;
    /* Start ADC1 calibration */
    ADC_StartCalibration(ADC);
    /* Check the end of ADC1 calibration */
    while (ADC_GetCalibrationStatus(ADC))
        ;
}
/**
 * @brief  Configures the different system clocks.
 */
void RCC_Configuration(void)
{
    /* Enable peripheral clocks ------------------------------------------------*/

    /* Enable GPIOC clocks */
    RCC_EnableAPB2PeriphClk(RCC_APB2_PERIPH_GPIOA | RCC_APB2_PERIPH_GPIOB, ENABLE);
    /* Enable ADC clocks */
    RCC_EnableAHBPeriphClk(RCC_AHB_PERIPH_ADC, ENABLE);

    /* RCC_ADCHCLK_DIV16*/
    ADC_ConfigClk(ADC_CTRL3_CKMOD_AHB, RCC_ADCHCLK_DIV16);
    RCC_ConfigAdc1mClk(RCC_ADC1MCLK_SRC_HSE, RCC_ADC1MCLK_DIV8); // selsect HSE as RCC ADC1M CLK Source
}

/**
 * @brief  Configures the different GPIO ports.
 */
void GPIO_Configuration(void)
{
    GPIO_InitType GPIO_InitStructure;

    GPIO_InitStruct(&GPIO_InitStructure);
    /* Configure PC0 PC1 as analog input -------------------------*/
    GPIO_InitStructure.Pin = GPIO_PIN_5 | GPIO_PIN_6 | GPIO_PIN_7;
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Analog;
    GPIO_InitPeripheral(GPIOA, &GPIO_InitStructure);

    GPIO_InitStruct(&GPIO_InitStructure);
    /* Configure PC0 PC1 as analog input -------------------------*/
    GPIO_InitStructure.Pin = GPIO_PIN_0 | GPIO_PIN_1;
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Analog;
    GPIO_InitPeripheral(GPIOB, &GPIO_InitStructure);

    GPIO_InitStruct(&GPIO_InitStructure);
    /* Configure PC0 PC1 as analog input -------------------------*/
    GPIO_InitStructure.Pin = GPIO_PIN_6;
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
    GPIO_InitPeripheral(GPIOB, &GPIO_InitStructure);
}

uint16_t ADC_GetData(uint8_t ADC_Channel)
{
    uint16_t dat;
    ADC_ConfigRegularChannel(ADC, ADC_Channel, 1, ADC_SAMP_TIME_55CYCLES5);
    /* Start ADC Software Conversion */
    ADC_EnableSoftwareStartConv(ADC, ENABLE);
    while (ADC_GetFlagStatus(ADC, ADC_FLAG_ENDC) == 0)
    {
    }
    ADC_ClearFlag(ADC, ADC_FLAG_ENDC);
    ADC_ClearFlag(ADC, ADC_FLAG_STR);
    dat = ADC_GetDat(ADC);
    return dat;
}
/**
 * @brief  Main program.
 */
int main(void)
{
    /*!< At this stage the microcontroller clock setting is already configured,
         this is done through SystemInit() function which is called from startup
         file (startup_n32l43x_xx.s) before to branch to application main.
         To reconfigure the default setting of SystemInit() function, refer to
         system_n32l43x.c file
       */
    /* Initialize USART,TX: PA9 RX: PA10*/
    log_init();
    log_info("\r\n MCU Reset!\r\n");
    LEDInit(LED1_PORT, LED1_PIN);
    LEDOff(LED1_PORT, LED1_PIN);
    /* Enable PWR Clock */
    RCC_EnableAPB1PeriphClk(RCC_APB1_PERIPH_PWR, ENABLE);

    /* System clocks configuration ---------------------------------------------*/
    RCC_Configuration();

    /* GPIO configuration ------------------------------------------------------*/
    GPIO_Configuration();
    ADC_Initial();

    DBG_ConfigPeriph(DBG_SLEEP, ENABLE);
    while (1)
    {
        /* Insert a long delay */
        log_info("\r\n MCU Prepare Enter Low Power Run Mode Sysclock From MSI(4MHz)\r\n");
        Delay_ms(2000);

        ADCConvertedValue[0] = ADC_GetData(ADC_CH_9_PB0);
        ADCConvertedValue[1] = ADC_GetData(ADC_CH_10_PB1);
        ADCConvertedValue[2] = ADC_GetData(ADC_CH_6_PA5);
        ADCConvertedValue[3] = ADC_GetData(ADC_CH_7_PA6);
        ADCConvertedValue[4] = ADC_GetData(ADC_CH_8_PA7);

        log_info("\r\n ADCConvertedValue[0] = %d \r\n", ADCConvertedValue[0]);
        log_info("\r\n ADCConvertedValue[1] = %d \r\n", ADCConvertedValue[1]);
        log_info("\r\n ADCConvertedValue[2] = %d \r\n", ADCConvertedValue[2]);
        log_info("\r\n ADCConvertedValue[3] = %d \r\n", ADCConvertedValue[3]);
        log_info("\r\n ADCConvertedValue[4] = %d \r\n", ADCConvertedValue[4]);

        LEDBlink(LED1_PORT, LED1_PIN);
        /* Request to enter LP RUN mode sysclock switch to MSI*/
        PWR_EnterLowPowerRunMode();
        Delay_ms(10);
        /* Exit LP RUN mode*/
        PWR_ExitLowPowerRunMode();

        //		ADCConvertedValue[0]=ADC_GetData(ADC_CH_9_PB0);
        //		ADCConvertedValue[1]=ADC_GetData(ADC_CH_10_PB1);
        //
        //		ADCConvertedValue[2]=ADC_GetData(ADC_CH_6_PA5);
        //		ADCConvertedValue[3]=ADC_GetData(ADC_CH_7_PA6);
        //		ADCConvertedValue[4]=ADC_GetData(ADC_CH_8_PA7);

        /*Switch sysclok from MSI to PLL*/
        SetSysClockToPLL(SystemCoreClock, SYSCLK_PLLSRC_HSE_PLLDIV2);
        log_init();
        log_info("\r\n MCU Run In Run Mode Sysclock From PLL(64MHz)\r\n");

        ADCConvertedValue[0] = ADC_GetData(ADC_CH_9_PB0);
        ADCConvertedValue[1] = ADC_GetData(ADC_CH_10_PB1);

        ADCConvertedValue[2] = ADC_GetData(ADC_CH_6_PA5);
        ADCConvertedValue[3] = ADC_GetData(ADC_CH_7_PA6);
        ADCConvertedValue[4] = ADC_GetData(ADC_CH_8_PA7);

        log_info("\r\n ADCConvertedValue[0] = %d \r\n", ADCConvertedValue[0]);
        log_info("\r\n ADCConvertedValue[1] = %d \r\n", ADCConvertedValue[1]);
        log_info("\r\n ADCConvertedValue[2] = %d \r\n", ADCConvertedValue[2]);
        log_info("\r\n ADCConvertedValue[3] = %d \r\n", ADCConvertedValue[3]);
        log_info("\r\n ADCConvertedValue[4] = %d \r\n", ADCConvertedValue[4]);

        LEDBlink(LED1_PORT, LED1_PIN);
    }
}
